const express = require('express');
const sqlite3 = require('sqlite3').verbose();
const bodyParser = require('body-parser');
const session = require('express-session');
const bcrypt = require('bcrypt');
const path = require('path');
const PDFDocument = require('pdfkit');
const puppeteer = require('puppeteer');

const app = express();
const PORT = process.env.PORT || 3000;

// Middleware
app.use(bodyParser.json());
app.use(bodyParser.urlencoded({ extended: true }));
app.use(express.static(path.join(__dirname, '../public')));
app.use(session({
    secret: 'nexuship-secret-key-2024',
    resave: false,
    saveUninitialized: true,
    cookie: { secure: false }
}));

// Initialize Database
const db = new sqlite3.Database('./nexuship.db', (err) => {
    if (err) {
        console.error('Error opening database:', err);
    } else {
        console.log('Connected to SQLite database');
        initializeDatabase();
    }
});

// Initialize Database Tables
function initializeDatabase() {
    // Create shipments table
    db.run(`CREATE TABLE IF NOT EXISTS shipments (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        tracking_number TEXT UNIQUE NOT NULL,
        shipper_name TEXT,
        shipper_address TEXT,
        receiver_name TEXT,
        receiver_address TEXT,
        ship_date TEXT,
        delivery_date TEXT,
        current_status TEXT DEFAULT 'Pending',
        weight TEXT,
        service_type TEXT DEFAULT 'Express',
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )`);

    // Create tracking_history table
    db.run(`CREATE TABLE IF NOT EXISTS tracking_history (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        tracking_number TEXT NOT NULL,
        status TEXT NOT NULL,
        location TEXT NOT NULL,
        date_time TEXT NOT NULL,
        FOREIGN KEY (tracking_number) REFERENCES shipments(tracking_number)
    )`);

    // Create admin user (default password: Adedapo1997$)
    const defaultPassword = bcrypt.hashSync('Adedapo1997$', 10);
    
    db.run(`CREATE TABLE IF NOT EXISTS admins (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        username TEXT UNIQUE NOT NULL,
        password TEXT NOT NULL,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )`, () => {
        // Check if admin exists, if not create one
        db.get("SELECT * FROM admins WHERE username = 'admin'", (err, row) => {
            if (!row) {
                db.run("INSERT INTO admins (username, password) VALUES (?, ?)", 
                    ['admin', defaultPassword], (err) => {
                    if (!err) {
                        console.log('Default admin user created');
                    }
                });
            }
        });
    });
}

// Helper Functions
function generateTrackingNumber(count = 1) {
    const trackingNumbers = [];
    const prefix = 'NS';
    const timestamp = Date.now().toString().slice(-6);
    
    for (let i = 0; i < count; i++) {
        const randomNum = Math.floor(Math.random() * 9000000000) + 1000000000;
        trackingNumbers.push(`${prefix}${timestamp}${randomNum}`);
    }
    
    return trackingNumbers;
}

// API Routes

// Track Package
app.get('/api/track/:trackingNumber', (req, res) => {
    const trackingNumber = req.params.trackingNumber;
    
    db.get("SELECT * FROM shipments WHERE tracking_number = ?", [trackingNumber], (err, row) => {
        if (err) {
            return res.status(500).json({ error: 'Database error' });
        }
        
        if (!row) {
            return res.json({ found: false });
        }
        
        // Get tracking history
        db.all("SELECT * FROM tracking_history WHERE tracking_number = ? ORDER BY date_time DESC", 
            [trackingNumber], (err, history) => {
            if (err) {
                history = [];
            }
            
            res.json({
                found: true,
                ...row,
                history: history
            });
        });
    });
});

// Admin Login
app.post('/api/admin/login', (req, res) => {
    const { username, password } = req.body;
    
    db.get("SELECT * FROM admins WHERE username = ?", [username], (err, row) => {
        if (err) {
            return res.status(500).json({ success: false, message: 'Database error' });
        }
        
        if (!row) {
            return res.json({ success: false, message: 'Invalid credentials' });
        }
        
        bcrypt.compare(password, row.password, (err, result) => {
            if (result) {
                req.session.adminId = row.id;
                res.json({ 
                    success: true, 
                    token: 'session-' + Date.now(),
                    message: 'Login successful'
                });
            } else {
                res.json({ success: false, message: 'Invalid credentials' });
            }
        });
    });
});

// Admin Dashboard
app.get('/api/admin/dashboard', (req, res) => {
    db.all("SELECT current_status, COUNT(*) as count FROM shipments GROUP BY current_status", 
        (err, rows) => {
        if (err) {
            return res.status(500).json({ error: 'Database error' });
        }
        
        const stats = {
            total: 0,
            inTransit: 0,
            delivered: 0,
            pending: 0,
            outForDelivery: 0,
            onHold: 0
        };
        
        rows.forEach(row => {
            stats.total += row.count;
            if (row.current_status === 'In Transit') stats.inTransit = row.count;
            else if (row.current_status === 'Delivered') stats.delivered = row.count;
            else if (row.current_status === 'Pending') stats.pending = row.count;
            else if (row.current_status === 'Out for Delivery') stats.outForDelivery = row.count;
            else if (row.current_status === 'On Hold') stats.onHold = row.count;
        });
        
        res.json(stats);
    });
});

// Generate Tracking Numbers
app.post('/api/admin/generate', (req, res) => {
    const { count = 300 } = req.body;
    const trackingNumbers = generateTrackingNumber(count);
    
    let inserted = 0;
    let failed = 0;
    
    trackingNumbers.forEach(trackingNumber => {
        const shipDate = new Date().toISOString().split('T')[0];
        const deliveryDate = new Date(Date.now() + 7 * 24 * 60 * 60 * 1000).toISOString().split('T')[0];
        
        db.run(`INSERT INTO shipments (tracking_number, ship_date, delivery_date, current_status) 
                VALUES (?, ?, ?, 'Pending')`, 
            [trackingNumber, shipDate, deliveryDate], (err) => {
            if (err) {
                failed++;
            } else {
                inserted++;
            }
            
            // Check if all processed
            if (inserted + failed === trackingNumbers.length) {
                res.json({
                    success: true,
                    message: `Generated ${inserted} tracking numbers successfully`,
                    count: inserted,
                    startNumber: trackingNumbers[0],
                    endNumber: trackingNumbers[trackingNumbers.length - 1]
                });
            }
        });
    });
});

// Get All Shipments
app.get('/api/admin/shipments', (req, res) => {
    db.all("SELECT * FROM shipments ORDER BY created_at DESC LIMIT 100", (err, rows) => {
        if (err) {
            return res.status(500).json({ error: 'Database error' });
        }
        res.json({ shipments: rows });
    });
});

// Get Single Shipment
app.get('/api/admin/shipment/:trackingNumber', (req, res) => {
    const trackingNumber = req.params.trackingNumber;
    
    db.get("SELECT * FROM shipments WHERE tracking_number = ?", [trackingNumber], (err, row) => {
        if (err) {
            return res.status(500).json({ error: 'Database error' });
        }
        
        if (!row) {
            return res.status(404).json({ error: 'Shipment not found' });
        }
        
        res.json({ shipment: row });
    });
});

// Update Shipment
app.put('/api/admin/shipment/:trackingNumber', (req, res) => {
    const trackingNumber = req.params.trackingNumber;
    const { shipper_name, shipper_address, receiver_name, receiver_address, ship_date, delivery_date, weight } = req.body;
    
    db.run(`UPDATE shipments SET 
        shipper_name = ?,
        shipper_address = ?,
        receiver_name = ?,
        receiver_address = ?,
        ship_date = ?,
        delivery_date = ?,
        weight = ?,
        updated_at = CURRENT_TIMESTAMP
        WHERE tracking_number = ?`,
        [shipper_name, shipper_address, receiver_name, receiver_address, ship_date, delivery_date, weight, trackingNumber],
        (err) => {
            if (err) {
                return res.status(500).json({ error: 'Failed to update shipment' });
            }
            res.json({ success: true, message: 'Shipment updated successfully' });
        }
    );
});

// Update Status
app.post('/api/admin/update-status', (req, res) => {
    const { tracking_number, status, location, date, time } = req.body;
    
    const dateTime = `${date} ${time}`;
    
    // Update shipment status
    db.run(`UPDATE shipments SET 
        current_status = ?,
        updated_at = CURRENT_TIMESTAMP
        WHERE tracking_number = ?`,
        [status, tracking_number], (err) => {
        if (err) {
            return res.status(500).json({ error: 'Failed to update status' });
        }
        
        // Add to tracking history
        db.run(`INSERT INTO tracking_history (tracking_number, status, location, date_time) 
                VALUES (?, ?, ?, ?)`,
            [tracking_number, status, location, dateTime], (err) => {
            if (err) {
                console.error('Error adding to tracking history:', err);
            }
            
            res.json({ success: true, message: 'Status updated successfully' });
        });
    });
});

// Bulk Update Status
app.post('/api/admin/bulk-update', (req, res) => {
    const { status, location, date, time } = req.body;
    const dateTime = `${date} ${time}`;
    
    // Get all tracking numbers
    db.all("SELECT tracking_number FROM shipments", (err, rows) => {
        if (err) {
            return res.status(500).json({ error: 'Database error' });
        }
        
        let updated = 0;
        const trackingNumbers = rows.map(row => row.tracking_number);
        
        trackingNumbers.forEach(trackingNumber => {
            // Update shipment status
            db.run(`UPDATE shipments SET 
                current_status = ?,
                updated_at = CURRENT_TIMESTAMP
                WHERE tracking_number = ?`,
                [status, trackingNumber]);
            
            // Add to tracking history
            db.run(`INSERT INTO tracking_history (tracking_number, status, location, date_time) 
                    VALUES (?, ?, ?, ?)`,
                [trackingNumber, status, location, dateTime], (err) => {
                if (!err) {
                    updated++;
                }
                
                // Check if all updated
                if (updated === trackingNumbers.length) {
                    res.json({ 
                        success: true, 
                        message: 'Bulk update completed',
                        updated: updated
                    });
                }
            });
        });
    });
});

// Get Receipt Data
app.get('/api/receipt/:trackingNumber', (req, res) => {
    const trackingNumber = req.params.trackingNumber;
    
    db.get("SELECT * FROM shipments WHERE tracking_number = ?", [trackingNumber], (err, row) => {
        if (err || !row) {
            return res.status(404).json({ error: 'Shipment not found' });
        }
        res.json(row);
    });
});

// Generate PDF Receipt
app.get('/api/pdf/:trackingNumber', async (req, res) => {
    const trackingNumber = req.params.trackingNumber;
    
    db.get("SELECT * FROM shipments WHERE tracking_number = ?", [trackingNumber], async (err, shipment) => {
        if (err || !shipment) {
            return res.status(404).send('Shipment not found');
        }
        
        try {
            const doc = new PDFDocument();
            
            res.setHeader('Content-Type', 'application/pdf');
            res.setHeader('Content-Disposition', `attachment; filename=receipt-${trackingNumber}.pdf`);
            
            doc.pipe(res);
            
            // Header
            doc.fontSize(24).fillColor('#4D148C').text('NexusShip Receipt', { align: 'center' });
            doc.moveDown();
            
            doc.fontSize(12).fillColor('#333');
            doc.text(`Tracking Number: ${shipment.tracking_number}`, { align: 'center' });
            doc.text(`Generated: ${new Date().toLocaleDateString()}`, { align: 'center' });
            doc.moveDown();
            
            // Status
            doc.rect(50, 150, 500, 40).fill('#4D148C');
            doc.fillColor('white').fontSize(16).text(`Status: ${shipment.current_status}`, 60, 165);
            
            doc.moveDown(2);
            
            // Shipper Info
            doc.fontSize(14).fillColor('#4D148C').text('Shipper Information');
            doc.fontSize(12).fillColor('#333');
            doc.text(`Name: ${shipment.shipper_name || 'N/A'}`);
            doc.text(`Address: ${shipment.shipper_address || 'N/A'}`);
            doc.moveDown();
            
            // Receiver Info
            doc.fontSize(14).fillColor('#4D148C').text('Receiver Information');
            doc.fontSize(12).fillColor('#333');
            doc.text(`Name: ${shipment.receiver_name || 'N/A'}`);
            doc.text(`Address: ${shipment.receiver_address || 'N/A'}`);
            doc.moveDown();
            
            // Shipment Details
            doc.fontSize(14).fillColor('#4D148C').text('Shipment Details');
            doc.fontSize(12).fillColor('#333');
            doc.text(`Ship Date: ${shipment.ship_date || 'N/A'}`);
            doc.text(`Expected Delivery: ${shipment.delivery_date || 'N/A'}`);
            doc.text(`Weight: ${shipment.weight || 'N/A'}`);
            doc.text(`Service: ${shipment.service_type || 'Express'}`);
            
            doc.moveDown(2);
            
            // Footer
            doc.fontSize(10).fillColor('#666').text('Thank you for choosing NexusShip!', { align: 'center' });
            doc.text('For inquiries, contact support@nexuship.com', { align: 'center' });
            
            doc.end();
        } catch (error) {
            console.error('PDF generation error:', error);
            res.status(500).send('Failed to generate PDF');
        }
    });
});

// Generate Image Receipt
app.get('/api/image/:trackingNumber', async (req, res) => {
    const trackingNumber = req.params.trackingNumber;
    
    db.get("SELECT * FROM shipments WHERE tracking_number = ?", [trackingNumber], async (err, shipment) => {
        if (err || !shipment) {
            return res.status(404).send('Shipment not found');
        }
        
        try {
            const browser = await puppeteer.launch();
            const page = await browser.newPage();
            
            const html = `
                <!DOCTYPE html>
                <html>
                <head>
                    <style>
                        body { font-family: Arial, sans-serif; max-width: 800px; margin: 20px auto; padding: 20px; background: #f5f5f5; }
                        .receipt { background: white; padding: 40px; border-radius: 10px; box-shadow: 0 5px 15px rgba(0,0,0,0.1); }
                        .header { text-align: center; border-bottom: 3px solid #4D148C; padding-bottom: 20px; margin-bottom: 30px; }
                        .header h1 { color: #4D148C; margin: 0; font-size: 28px; }
                        .status { background: #4D148C; color: white; text-align: center; padding: 15px; border-radius: 5px; margin-bottom: 30px; font-size: 18px; }
                        .section { margin-bottom: 25px; }
                        .section h3 { color: #4D148C; margin: 0 0 10px 0; font-size: 16px; }
                        .section p { margin: 5px 0; color: #333; }
                        .footer { text-align: center; margin-top: 40px; color: #666; font-size: 12px; }
                    </style>
                </head>
                <body>
                    <div class="receipt">
                        <div class="header">
                            <h1>NexusShip Receipt</h1>
                            <p>Tracking Number: ${shipment.tracking_number}</p>
                            <p>Generated: ${new Date().toLocaleDateString()}</p>
                        </div>
                        <div class="status">
                            <strong>Status: ${shipment.current_status}</strong>
                        </div>
                        <div class="section">
                            <h3>Shipper Information</h3>
                            <p><strong>Name:</strong> ${shipment.shipper_name || 'N/A'}</p>
                            <p><strong>Address:</strong> ${shipment.shipper_address || 'N/A'}</p>
                        </div>
                        <div class="section">
                            <h3>Receiver Information</h3>
                            <p><strong>Name:</strong> ${shipment.receiver_name || 'N/A'}</p>
                            <p><strong>Address:</strong> ${shipment.receiver_address || 'N/A'}</p>
                        </div>
                        <div class="section">
                            <h3>Shipment Details</h3>
                            <p><strong>Ship Date:</strong> ${shipment.ship_date || 'N/A'}</p>
                            <p><strong>Expected Delivery:</strong> ${shipment.delivery_date || 'N/A'}</p>
                            <p><strong>Weight:</strong> ${shipment.weight || 'N/A'}</p>
                            <p><strong>Service:</strong> ${shipment.service_type || 'Express'}</p>
                        </div>
                        <div class="footer">
                            <p>Thank you for choosing NexusShip!</p>
                            <p>For inquiries, contact support@nexuship.com</p>
                        </div>
                    </div>
                </body>
                </html>
            `;
            
            await page.setContent(html);
            const screenshot = await page.screenshot({ type: 'png', fullPage: true });
            
            await browser.close();
            
            res.setHeader('Content-Type', 'image/png');
            res.setHeader('Content-Disposition', `attachment; filename=receipt-${trackingNumber}.png`);
            res.send(screenshot);
        } catch (error) {
            console.error('Image generation error:', error);
            res.status(500).send('Failed to generate image');
        }
    });
});

// Serve Admin Dashboard
app.get('/admin-dashboard.html', (req, res) => {
    // Simple session check
    if (!req.session.adminId) {
        return res.redirect('/admin.html');
    }
    res.sendFile(path.join(__dirname, '../public/admin-dashboard.html'));
});

// Root Route
app.get('/', (req, res) => {
    res.sendFile(path.join(__dirname, '../public/index.html'));
});

// Start Server
app.listen(PORT, () => {
    console.log(`NexusShip server running on port ${PORT}`);
    console.log(`Visit http://localhost:${PORT} to access the website`);
    console.log(`Admin login: http://localhost:${PORT}/admin.html`);
    console.log(`Default admin password: Adedapo1997$`);
});