// NexusShip Main JavaScript

// Track Package Function
async function trackPackage() {
    const trackingNumber = document.getElementById('trackingNumber').value.trim();
    
    if (!trackingNumber) {
        alert('Please enter a tracking number');
        return;
    }

    try {
        const response = await fetch(`/api/track/${trackingNumber}`);
        const data = await response.json();

        if (response.ok && data.found) {
            displayTrackingResults(data);
        } else {
            displayTrackingNotFound();
        }
    } catch (error) {
        console.error('Error tracking package:', error);
        alert('An error occurred while tracking your package. Please try again.');
    }
}

// Display Tracking Results
function displayTrackingResults(data) {
    const resultsDiv = document.getElementById('trackingResults');
    
    let statusClass = 'pending';
    if (data.current_status === 'In Transit') statusClass = 'in-transit';
    else if (data.current_status === 'Out for Delivery') statusClass = 'out-for-delivery';
    else if (data.current_status === 'Delivered') statusClass = 'delivered';
    else if (data.current_status === 'On Hold') statusClass = 'on-hold';

    let historyHTML = '';
    if (data.history && data.history.length > 0) {
        historyHTML = data.history.map((item, index) => `
            <div class="timeline-item ${index === 0 ? 'active' : ''}">
                <div class="timeline-icon">
                    ${index === 0 ? '📦' : '✓'}
                </div>
                <div class="timeline-content">
                    <div class="timeline-status">${item.status}</div>
                    <div class="timeline-location">${item.location}</div>
                    <div class="timeline-date-time">${formatDateTime(item.date_time)}</div>
                </div>
            </div>
        `).join('');
    } else {
        historyHTML = `
            <div class="timeline-item active">
                <div class="timeline-icon">📦</div>
                <div class="timeline-content">
                    <div class="timeline-status">${data.current_status}</div>
                    <div class="timeline-location">Processing Center</div>
                    <div class="timeline-date-time">${formatDateTime(new Date())}</div>
                </div>
            </div>
        `;
    }

    resultsDiv.innerHTML = `
        <div class="tracking-header">
            <div class="tracking-number">Tracking #${data.tracking_number}</div>
            <span class="status-badge ${statusClass}">${data.current_status}</span>
            
            <div class="shipment-details">
                <div class="detail-box">
                    <label>Shipper</label>
                    <span>${data.shipper_name}</span>
                </div>
                <div class="detail-box">
                    <label>Shipper Address</label>
                    <span>${data.shipper_address}</span>
                </div>
                <div class="detail-box">
                    <label>Receiver</label>
                    <span>${data.receiver_name}</span>
                </div>
                <div class="detail-box">
                    <label>Receiver Address</label>
                    <span>${data.receiver_address}</span>
                </div>
                <div class="detail-box">
                    <label>Ship Date</label>
                    <span>${formatDate(data.ship_date)}</span>
                </div>
                <div class="detail-box">
                    <label>Expected Delivery</label>
                    <span>${formatDate(data.delivery_date)}</span>
                </div>
            </div>
        </div>
        
        <div class="tracking-timeline">
            <h3 style="color: var(--fedex-purple); margin-bottom: 20px;">Tracking History</h3>
            ${historyHTML}
        </div>

        <div style="text-align: center; margin-top: 30px;">
            <button class="btn btn-primary" onclick="printReceipt('${data.tracking_number}')">Print Receipt</button>
            <button class="btn btn-success" onclick="downloadPDF('${data.tracking_number}')">Download PDF</button>
            <button class="btn btn-warning" onclick="downloadImage('${data.tracking_number}')">Download Image</button>
        </div>
    `;
    
    resultsDiv.classList.add('active');
}

// Display Tracking Not Found
function displayTrackingNotFound() {
    const resultsDiv = document.getElementById('trackingResults');
    resultsDiv.innerHTML = `
        <div style="text-align: center; padding: 40px;">
            <div style="font-size: 60px; margin-bottom: 20px;">❌</div>
            <h2 style="color: var(--fedex-purple); margin-bottom: 10px;">Tracking Number Not Found</h2>
            <p style="color: #666;">The tracking number you entered is not valid or does not exist in our system. Please check the number and try again.</p>
        </div>
    `;
    resultsDiv.classList.add('active');
}

// Print Receipt
async function printReceipt(trackingNumber) {
    try {
        const response = await fetch(`/api/receipt/${trackingNumber}`);
        const data = await response.json();
        
        if (response.ok) {
            const printWindow = window.open('', '_blank');
            printWindow.document.write(`
                <html>
                <head>
                    <title>Receipt - ${trackingNumber}</title>
                    <style>
                        body { font-family: Arial, sans-serif; max-width: 800px; margin: 20px auto; padding: 20px; }
                        .header { text-align: center; border-bottom: 3px solid #4D148C; padding-bottom: 20px; margin-bottom: 30px; }
                        .header h1 { color: #4D148C; margin: 0; }
                        .details { display: grid; grid-template-columns: 1fr 1fr; gap: 30px; margin-bottom: 30px; }
                        .box { background: #f5f5f5; padding: 20px; border-radius: 5px; }
                        .box h3 { color: #4D148C; margin: 0 0 15px 0; }
                        .box p { margin: 5px 0; }
                        .status { text-align: center; padding: 15px; background: #4D148C; color: white; border-radius: 5px; margin-bottom: 20px; }
                        .footer { text-align: center; margin-top: 40px; color: #666; }
                    </style>
                </head>
                <body>
                    <div class="header">
                        <h1>NexusShip Receipt</h1>
                        <p>Tracking Number: ${trackingNumber}</p>
                    </div>
                    <div class="status">
                        <strong>Status: ${data.current_status}</strong>
                    </div>
                    <div class="details">
                        <div class="box">
                            <h3>Shipper Information</h3>
                            <p><strong>Name:</strong> ${data.shipper_name}</p>
                            <p><strong>Address:</strong> ${data.shipper_address}</p>
                        </div>
                        <div class="box">
                            <h3>Receiver Information</h3>
                            <p><strong>Name:</strong> ${data.receiver_name}</p>
                            <p><strong>Address:</strong> ${data.receiver_address}</p>
                        </div>
                    </div>
                    <div class="box">
                        <h3>Shipment Details</h3>
                        <p><strong>Ship Date:</strong> ${formatDate(data.ship_date)}</p>
                        <p><strong>Expected Delivery:</strong> ${formatDate(data.delivery_date)}</p>
                        <p><strong>Weight:</strong> ${data.weight || 'N/A'}</p>
                        <p><strong>Service:</strong> ${data.service_type || 'Express'}</p>
                    </div>
                    <div class="footer">
                        <p>Receipt generated on ${new Date().toLocaleDateString()}</p>
                        <p>Thank you for choosing NexusShip!</p>
                    </div>
                </body>
                </html>
            `);
            printWindow.document.close();
            printWindow.print();
        }
    } catch (error) {
        console.error('Error printing receipt:', error);
        alert('Failed to generate receipt');
    }
}

// Download PDF
async function downloadPDF(trackingNumber) {
    try {
        window.open(`/api/pdf/${trackingNumber}`, '_blank');
    } catch (error) {
        console.error('Error downloading PDF:', error);
        alert('Failed to download PDF');
    }
}

// Download Image
async function downloadImage(trackingNumber) {
    try {
        window.open(`/api/image/${trackingNumber}`, '_blank');
    } catch (error) {
        console.error('Error downloading image:', error);
        alert('Failed to download image');
    }
}

// Format Date
function formatDate(dateString) {
    if (!dateString) return 'N/A';
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'long',
        day: 'numeric'
    });
}

// Format Date Time
function formatDateTime(dateString) {
    if (!dateString) return 'N/A';
    const date = new Date(dateString);
    return date.toLocaleString('en-US', {
        year: 'numeric',
        month: 'short',
        day: 'numeric',
        hour: '2-digit',
        minute: '2-digit'
    });
}

// Contact Form Handler
document.addEventListener('DOMContentLoaded', function() {
    const contactForm = document.getElementById('contactForm');
    if (contactForm) {
        contactForm.addEventListener('submit', function(e) {
            e.preventDefault();
            alert('Thank you for contacting us! We will respond shortly.');
            contactForm.reset();
        });
    }
});