// NexusShip Admin JavaScript

// Check authentication on page load
document.addEventListener('DOMContentLoaded', function() {
    const token = localStorage.getItem('adminToken');
    if (!token) {
        window.location.href = '/admin.html';
        return;
    }
    
    // Set default date/time values
    setDefaultDateTime();
    
    // Load initial dashboard data
    loadDashboard();
});

// Set default date/time to current
function setDefaultDateTime() {
    const now = new Date();
    const dateStr = now.toISOString().split('T')[0];
    const timeStr = now.toTimeString().split(' ')[0].substring(0, 5);
    
    document.getElementById('updateDate').value = dateStr;
    document.getElementById('updateTime').value = timeStr;
    document.getElementById('bulkDate').value = dateStr;
    document.getElementById('bulkTime').value = timeStr;
}

// Show Section
function showSection(sectionName) {
    // Hide all sections
    document.querySelectorAll('.admin-section').forEach(section => {
        section.style.display = 'none';
    });
    
    // Show selected section
    document.getElementById(`${sectionName}-section`).style.display = 'block';
    
    // Update active menu
    document.querySelectorAll('.admin-menu a').forEach(link => {
        link.classList.remove('active');
    });
    event.target.classList.add('active');
}

// Load Dashboard
async function loadDashboard() {
    try {
        const response = await fetch('/api/admin/dashboard');
        const data = await response.json();
        
        if (response.ok) {
            document.getElementById('totalShipments').textContent = data.total || 0;
            document.getElementById('inTransit').textContent = data.inTransit || 0;
            document.getElementById('delivered').textContent = data.delivered || 0;
            document.getElementById('pending').textContent = data.pending || 0;
        }
    } catch (error) {
        console.error('Error loading dashboard:', error);
    }
}

// Generate Tracking Numbers
async function generateTrackingNumbers() {
    const count = document.getElementById('generateCount').value || 300;
    
    try {
        const response = await fetch('/api/admin/generate', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${localStorage.getItem('adminToken')}`
            },
            body: JSON.stringify({ count: parseInt(count) })
        });
        
        const data = await response.json();
        
        if (response.ok) {
            document.getElementById('generateResult').innerHTML = `
                <div style="background: #d4edda; color: #155724; padding: 20px; border-radius: 5px; border: 1px solid #c3e6cb;">
                    <h3>✓ Success!</h3>
                    <p>${data.message}</p>
                    <p>Generated ${data.count} tracking numbers starting from: <strong>${data.startNumber}</strong></p>
                    <p>Ending with: <strong>${data.endNumber}</strong></p>
                </div>
            `;
            loadDashboard();
        } else {
            alert(data.message || 'Failed to generate tracking numbers');
        }
    } catch (error) {
        console.error('Error generating tracking numbers:', error);
        alert('An error occurred while generating tracking numbers');
    }
}

// Load Shipments
async function loadShipments() {
    try {
        const response = await fetch('/api/admin/shipments', {
            headers: {
                'Authorization': `Bearer ${localStorage.getItem('adminToken')}`
            }
        });
        
        const data = await response.json();
        
        if (response.ok && data.shipments) {
            displayShipments(data.shipments);
        }
    } catch (error) {
        console.error('Error loading shipments:', error);
        alert('Failed to load shipments');
    }
}

// Display Shipments
function displayShipments(shipments) {
    const tbody = document.getElementById('shipmentsBody');
    
    if (!shipments || shipments.length === 0) {
        tbody.innerHTML = '<tr><td colspan="6" style="text-align: center;">No shipments found</td></tr>';
        return;
    }
    
    tbody.innerHTML = shipments.map(shipment => `
        <tr>
            <td><strong>${shipment.tracking_number}</strong></td>
            <td>${shipment.shipper_name || 'N/A'}</td>
            <td>${shipment.receiver_name || 'N/A'}</td>
            <td><span class="status-badge ${getStatusClass(shipment.current_status)}">${shipment.current_status}</span></td>
            <td>${formatDate(shipment.ship_date)}</td>
            <td class="admin-actions">
                <button class="btn btn-primary" style="padding: 5px 10px; font-size: 12px;" onclick="editShipment('${shipment.tracking_number}')">Edit</button>
                <button class="btn btn-warning" style="padding: 5px 10px; font-size: 12px;" onclick="quickUpdate('${shipment.tracking_number}')">Update</button>
            </td>
        </tr>
    `).join('');
}

// Search Shipment
async function searchShipment() {
    const trackingNumber = document.getElementById('searchTracking').value.trim();
    
    if (!trackingNumber) {
        loadShipments();
        return;
    }
    
    try {
        const response = await fetch(`/api/admin/shipment/${trackingNumber}`, {
            headers: {
                'Authorization': `Bearer ${localStorage.getItem('adminToken')}`
            }
        });
        
        const data = await response.json();
        
        if (response.ok && data.shipment) {
            displayShipments([data.shipment]);
        } else {
            alert('Shipment not found');
            loadShipments();
        }
    } catch (error) {
        console.error('Error searching shipment:', error);
        alert('Failed to search shipment');
    }
}

// Edit Shipment
async function editShipment(trackingNumber) {
    try {
        const response = await fetch(`/api/admin/shipment/${trackingNumber}`, {
            headers: {
                'Authorization': `Bearer ${localStorage.getItem('adminToken')}`
            }
        });
        
        const data = await response.json();
        
        if (response.ok && data.shipment) {
            const shipment = data.shipment;
            
            document.getElementById('editTrackingNumber').value = shipment.tracking_number;
            document.getElementById('editShipperName').value = shipment.shipper_name || '';
            document.getElementById('editShipperAddress').value = shipment.shipper_address || '';
            document.getElementById('editReceiverName').value = shipment.receiver_name || '';
            document.getElementById('editReceiverAddress').value = shipment.receiver_address || '';
            document.getElementById('editShipDate').value = shipment.ship_date ? shipment.ship_date.split('T')[0] : '';
            document.getElementById('editDeliveryDate').value = shipment.delivery_date ? shipment.delivery_date.split('T')[0] : '';
            document.getElementById('editWeight').value = shipment.weight || '';
            
            document.getElementById('editModal').classList.add('active');
        }
    } catch (error) {
        console.error('Error loading shipment:', error);
        alert('Failed to load shipment details');
    }
}

// Close Modal
function closeModal() {
    document.getElementById('editModal').classList.remove('active');
}

// Handle Edit Form Submit
document.getElementById('editForm').addEventListener('submit', async function(e) {
    e.preventDefault();
    
    const trackingNumber = document.getElementById('editTrackingNumber').value;
    
    const shipmentData = {
        shipper_name: document.getElementById('editShipperName').value,
        shipper_address: document.getElementById('editShipperAddress').value,
        receiver_name: document.getElementById('editReceiverName').value,
        receiver_address: document.getElementById('editReceiverAddress').value,
        ship_date: document.getElementById('editShipDate').value,
        delivery_date: document.getElementById('editDeliveryDate').value,
        weight: document.getElementById('editWeight').value
    };
    
    try {
        const response = await fetch(`/api/admin/shipment/${trackingNumber}`, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${localStorage.getItem('adminToken')}`
            },
            body: JSON.stringify(shipmentData)
        });
        
        const data = await response.json();
        
        if (response.ok) {
            alert('Shipment updated successfully!');
            closeModal();
            loadShipments();
            loadDashboard();
        } else {
            alert(data.message || 'Failed to update shipment');
        }
    } catch (error) {
        console.error('Error updating shipment:', error);
        alert('An error occurred while updating shipment');
    }
});

// Update Status
async function updateStatus() {
    const trackingNumber = document.getElementById('updateTracking').value.trim();
    const status = document.getElementById('updateStatus').value;
    const location = document.getElementById('updateLocation').value.trim();
    const date = document.getElementById('updateDate').value;
    const time = document.getElementById('updateTime').value;
    
    if (!trackingNumber) {
        alert('Please enter a tracking number');
        return;
    }
    
    if (!location) {
        alert('Please enter a location');
        return;
    }
    
    try {
        const response = await fetch('/api/admin/update-status', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${localStorage.getItem('adminToken')}`
            },
            body: JSON.stringify({
                tracking_number: trackingNumber,
                status: status,
                location: location,
                date: date,
                time: time
            })
        });
        
        const data = await response.json();
        
        if (response.ok) {
            alert('Status updated successfully!');
            document.getElementById('updateTracking').value = '';
            document.getElementById('updateLocation').value = '';
            loadShipments();
            loadDashboard();
        } else {
            alert(data.message || 'Failed to update status');
        }
    } catch (error) {
        console.error('Error updating status:', error);
        alert('An error occurred while updating status');
    }
}

// Bulk Update
async function bulkUpdate() {
    if (!confirm('Are you sure you want to update ALL shipments? This action cannot be undone.')) {
        return;
    }
    
    const status = document.getElementById('bulkStatus').value;
    const location = document.getElementById('bulkLocation').value.trim();
    const date = document.getElementById('bulkDate').value;
    const time = document.getElementById('bulkTime').value;
    
    if (!location) {
        alert('Please enter a location');
        return;
    }
    
    try {
        const response = await fetch('/api/admin/bulk-update', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${localStorage.getItem('adminToken')}`
            },
            body: JSON.stringify({
                status: status,
                location: location,
                date: date,
                time: time
            })
        });
        
        const data = await response.json();
        
        if (response.ok) {
            alert(`Successfully updated ${data.updated} shipments!`);
            loadShipments();
            loadDashboard();
        } else {
            alert(data.message || 'Failed to bulk update');
        }
    } catch (error) {
        console.error('Error bulk updating:', error);
        alert('An error occurred during bulk update');
    }
}

// Quick Update
function quickUpdate(trackingNumber) {
    document.getElementById('updateTracking').value = trackingNumber;
    showSection('update');
}

// Logout
function logout() {
    localStorage.removeItem('adminToken');
    window.location.href = '/admin.html';
}

// Helper Functions
function getStatusClass(status) {
    if (status === 'In Transit') return 'in-transit';
    if (status === 'Out for Delivery') return 'out-for-delivery';
    if (status === 'Delivered') return 'delivered';
    if (status === 'On Hold') return 'on-hold';
    return 'pending';
}

function formatDate(dateString) {
    if (!dateString) return 'N/A';
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', {
        month: 'short',
        day: 'numeric',
        year: 'numeric'
    });
}