// NexusShip Main JavaScript

document.addEventListener('DOMContentLoaded', function() {
    // Initialize tracking functionality
    initTracking();
    
    // Initialize admin panel functionality
    initAdminPanel();
    
    // Initialize modals
    initModals();
});

// Tracking Functionality
function initTracking() {
    const trackingForm = document.getElementById('trackingForm');
    const trackingInput = document.getElementById('trackingNumber');
    const trackingButton = document.getElementById('trackButton');
    const trackingResults = document.getElementById('trackingResults');
    
    if (trackingForm) {
        trackingForm.addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const trackingNumber = trackingInput.value.trim();
            
            if (!trackingNumber) {
                showError('Please enter a tracking number');
                return;
            }
            
            // Show loading state
            trackingButton.innerHTML = '<span class="loading"></span> Tracking...';
            trackingButton.disabled = true;
            
            try {
                const response = await fetch('/api/track', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        tracking_number: trackingNumber
                    })
                });
                
                if (response.ok) {
                    const data = await response.json();
                    displayTrackingResults(data);
                } else {
                    const error = await response.json();
                    showError(error.error || 'Tracking number not found');
                }
            } catch (error) {
                showError('An error occurred while tracking. Please try again.');
            }
            
            // Reset button
            trackingButton.innerHTML = 'Track Package';
            trackingButton.disabled = false;
        });
    }
}

function displayTrackingResults(data) {
    const trackingResults = document.getElementById('trackingResults');
    const trackingInfo = document.querySelector('.tracking-info');
    const timeline = document.querySelector('.tracking-timeline');
    
    // Update tracking info
    trackingInfo.innerHTML = `
        <div class="info-box">
            <h4>Tracking Number</h4>
            <p>${data.tracking_number}</p>
        </div>
        <div class="info-box">
            <h4>Status</h4>
            <p><span class="status-badge status-${data.status.replace(/\s/g, '')}">${data.status}</span></p>
        </div>
        <div class="info-box">
            <h4>Shipper</h4>
            <p>${data.shipper_name}</p>
            <p style="font-size: 14px; margin-top: 5px;">${data.shipper_address}</p>
        </div>
        <div class="info-box">
            <h4>Receiver</h4>
            <p>${data.receiver_name}</p>
            <p style="font-size: 14px; margin-top: 5px;">${data.receiver_address}</p>
        </div>
        <div class="info-box">
            <h4>Shipment Date</h4>
            <p>${data.shipment_date}</p>
        </div>
        <div class="info-box">
            <h4>Expected Delivery</h4>
            <p>${data.delivery_date || 'Pending'}</p>
        </div>
        <div class="info-box">
            <h4>Service Type</h4>
            <p>${data.service_type}</p>
        </div>
        <div class="info-box">
            <h4>Weight</h4>
            <p>${data.weight} kg</p>
        </div>
    `;
    
    // Update timeline
    let timelineHTML = '';
    data.history.forEach((item, index) => {
        const icon = getTimelineIcon(item.status);
        timelineHTML += `
            <div class="timeline-item">
                <div class="timeline-icon">${icon}</div>
                <div class="timeline-content">
                    <h4>${item.status}</h4>
                    <p><strong>Location:</strong> ${item.location}</p>
                    <p>${item.description}</p>
                    <p class="timeline-date">${item.timestamp}</p>
                    <span class="timeline-backcode">Backcode: ${item.backcode}</span>
                </div>
            </div>
        `;
    });
    
    timeline.innerHTML = timelineHTML;
    
    // Show results
    trackingResults.classList.add('active');
    
    // Add receipt buttons
    addReceiptButtons(data.tracking_number);
}

function getTimelineIcon(status) {
    const icons = {
        'Processing': '📦',
        'In Transit': '🚚',
        'Out for Delivery': '📬',
        'Delivered': '✅',
        'On Hold': '⏸️',
        'Exception': '⚠️'
    };
    return icons[status] || '📋';
}

function addReceiptButtons(trackingNumber) {
    const timeline = document.querySelector('.tracking-timeline');
    
    const buttonsHTML = `
        <div class="receipt-actions" style="margin-top: 40px;">
            <button class="btn-success" onclick="downloadReceiptPDF('${trackingNumber}')">
                📄 Download PDF Receipt
            </button>
            <button class="btn-primary" onclick="printReceipt('${trackingNumber}')">
                🖨️ Print Receipt
            </button>
            <button class="btn-secondary" onclick="downloadReceiptImage('${trackingNumber}')">
                📷 Download as Image
            </button>
        </div>
    `;
    
    timeline.innerHTML += buttonsHTML;
}

async function downloadReceiptPDF(trackingNumber) {
    window.open(`/api/receipt/pdf/${trackingNumber}`, '_blank');
}

function printReceipt(trackingNumber) {
    // Open print dialog
    const url = `/api/receipt/pdf/${trackingNumber}`;
    const printWindow = window.open(url, '_blank');
    printWindow.onload = function() {
        printWindow.print();
    };
}

async function downloadReceiptImage(trackingNumber) {
    // For image download, we'll create a canvas from the receipt
    alert('Image download feature - A screenshot of the receipt will be generated. Please use your browser\'s screenshot functionality for now.');
    
    // Open receipt in new tab for screenshot
    window.open(`/api/receipt/pdf/${trackingNumber}`, '_blank');
}

function showError(message) {
    const trackingResults = document.getElementById('trackingResults');
    trackingResults.innerHTML = `
        <div class="alert alert-error">
            <strong>Error:</strong> ${message}
        </div>
    `;
    trackingResults.classList.add('active');
}

// Admin Panel Functionality
function initAdminPanel() {
    const generateButton = document.getElementById('generateTrackingNumbers');
    const shipmentsTable = document.getElementById('shipmentsTable');
    
    if (generateButton) {
        generateButton.addEventListener('click', async function() {
            if (confirm('Are you sure you want to generate 300 new tracking numbers?')) {
                await generateTrackingNumbers();
            }
        });
    }
    
    if (shipmentsTable) {
        loadShipments();
    }
    
    // Initialize status update form
    const updateStatusForm = document.getElementById('updateStatusForm');
    if (updateStatusForm) {
        updateStatusForm.addEventListener('submit', handleStatusUpdate);
    }
}

async function generateTrackingNumbers() {
    const generateButton = document.getElementById('generateTrackingNumbers');
    generateButton.innerHTML = '<span class="loading"></span> Generating...';
    generateButton.disabled = true;
    
    try {
        const response = await fetch('/api/admin/generate-tracking', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                count: 300
            })
        });
        
        const data = await response.json();
        
        if (data.tracking_numbers) {
            alert(`Successfully generated ${data.tracking_numbers.length} tracking numbers!`);
            loadShipments();
        }
    } catch (error) {
        alert('Error generating tracking numbers: ' + error.message);
    }
    
    generateButton.innerHTML = 'Generate 300 Tracking Numbers';
    generateButton.disabled = false;
}

async function loadShipments() {
    const tableBody = document.querySelector('#shipmentsTable tbody');
    
    try {
        const response = await fetch('/api/admin/shipments');
        const data = await response.json();
        
        if (data.shipments) {
            let html = '';
            data.shipments.forEach(shipment => {
                html += `
                    <tr>
                        <td>${shipment.tracking_number}</td>
                        <td>${shipment.shipper_name}</td>
                        <td>${shipment.receiver_name}</td>
                        <td><span class="status-badge status-${shipment.status.replace(/\s/g, '')}">${shipment.status}</span></td>
                        <td>${shipment.shipment_date}</td>
                        <td>${shipment.delivery_date || 'Pending'}</td>
                        <td>
                            <button class="btn-success" onclick="editShipment(${shipment.id})">Edit</button>
                            <button class="btn-danger" onclick="deleteShipment(${shipment.id})">Delete</button>
                            <button class="btn-secondary" onclick="updateStatus(${shipment.id}, '${shipment.tracking_number}')">Update Status</button>
                            <button class="btn-primary" onclick="downloadReceiptPDF('${shipment.tracking_number}')">Receipt</button>
                        </td>
                    </tr>
                `;
            });
            
            tableBody.innerHTML = html;
        }
    } catch (error) {
        tableBody.innerHTML = '<tr><td colspan="7" style="text-align: center;">Error loading shipments</td></tr>';
    }
}

function editShipment(id) {
    // This would open an edit modal
    alert(`Edit shipment with ID: ${id}. Full edit modal would be implemented here.`);
}

async function deleteShipment(id) {
    if (confirm('Are you sure you want to delete this shipment?')) {
        try {
            const response = await fetch(`/api/admin/shipment/${id}`, {
                method: 'DELETE'
            });
            
            if (response.ok) {
                alert('Shipment deleted successfully!');
                loadShipments();
            } else {
                alert('Error deleting shipment');
            }
        } catch (error) {
            alert('Error deleting shipment: ' + error.message);
        }
    }
}

function updateStatus(id, trackingNumber) {
    const modal = document.getElementById('statusModal');
    const trackingNumberInput = document.getElementById('statusTrackingNumber');
    const shipmentIdInput = document.getElementById('shipmentId');
    
    if (modal && trackingNumberInput && shipmentIdInput) {
        trackingNumberInput.value = trackingNumber;
        shipmentIdInput.value = id;
        modal.classList.add('active');
    }
}

async function handleStatusUpdate(e) {
    e.preventDefault();
    
    const trackingNumber = document.getElementById('statusTrackingNumber').value;
    const shipmentId = document.getElementById('shipmentId').value;
    const status = document.getElementById('newStatus').value;
    const location = document.getElementById('newLocation').value;
    const description = document.getElementById('newDescription').value;
    
    if (!location) {
        alert('Please enter a location for the status update');
        return;
    }
    
    try {
        const response = await fetch(`/api/admin/shipment/${shipmentId}`, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                status: status,
                location: location,
                description: description
            })
        });
        
        if (response.ok) {
            alert('Status updated successfully!');
            closeModal('statusModal');
            loadShipments();
        } else {
            alert('Error updating status');
        }
    } catch (error) {
        alert('Error updating status: ' + error.message);
    }
}

// Modal Functions
function initModals() {
    // Close modal when clicking outside
    window.addEventListener('click', function(e) {
        if (e.target.classList.contains('modal')) {
            e.target.classList.remove('active');
        }
    });
    
    // Close modal buttons
    const closeButtons = document.querySelectorAll('.modal-close');
    closeButtons.forEach(button => {
        button.addEventListener('click', function() {
            const modal = button.closest('.modal');
            modal.classList.remove('active');
        });
    });
}

function closeModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.classList.remove('active');
    }
}

// Bulk Status Update
async function bulkUpdateStatus() {
    const selectedCheckboxes = document.querySelectorAll('.shipment-checkbox:checked');
    
    if (selectedCheckboxes.length === 0) {
        alert('Please select at least one shipment to update');
        return;
    }
    
    const trackingNumbers = Array.from(selectedCheckboxes).map(cb => cb.dataset.tracking);
    const status = document.getElementById('bulkStatus').value;
    const location = document.getElementById('bulkLocation').value;
    const description = document.getElementById('bulkDescription').value;
    
    if (!location) {
        alert('Please enter a location for the status update');
        return;
    }
    
    try {
        const response = await fetch('/api/admin/bulk-update-status', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                tracking_numbers: trackingNumbers,
                status: status,
                location: location,
                description: description
            })
        });
        
        const data = await response.json();
        
        if (data.success) {
            alert(`Successfully updated ${data.updated} shipments!`);
            loadShipments();
        }
    } catch (error) {
        alert('Error updating shipments: ' + error.message);
    }
}

// Helper Functions
function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'short',
        day: 'numeric'
    });
}

function formatDateTime(dateString) {
    const date = new Date(dateString);
    return date.toLocaleString('en-US', {
        year: 'numeric',
        month: 'short',
        day: 'numeric',
        hour: '2-digit',
        minute: '2-digit'
    });
}

// Smooth scroll for navigation
document.querySelectorAll('a[href^="#"]').forEach(anchor => {
    anchor.addEventListener('click', function(e) {
        e.preventDefault();
        const target = document.querySelector(this.getAttribute('href'));
        if (target) {
            target.scrollIntoView({
                behavior: 'smooth'
            });
        }
    });
});

// Form validation
function validateEmail(email) {
    const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return re.test(email);
}

function validatePhone(phone) {
    const re = /^[\d\s\-\+\(\)]+$/;
    return re.test(phone);
}