from flask import Flask, render_template, request, jsonify, session, redirect, url_for, send_file
from flask_sqlalchemy import SQLAlchemy
from werkzeug.security import generate_password_hash, check_password_hash
from datetime import datetime
import random
import string
import os
from io import BytesIO
import base64
from reportlab.lib.pagesizes import letter, A4
from reportlab.lib import colors
from reportlab.platypus import SimpleDocTemplate, Table, TableStyle, Paragraph, Spacer, Image
from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
from reportlab.lib.enums import TA_CENTER, TA_LEFT
import qrcode
from PIL import Image as PILImage

app = Flask(__name__)
app.config['SECRET_KEY'] = 'nexusship-secret-key-2024'
app.config['SQLALCHEMY_DATABASE_URI'] = 'sqlite:///nexusship.db'
app.config['SQLALCHEMY_TRACK_MODIFICATIONS'] = False
app.config['MAX_CONTENT_LENGTH'] = 16 * 1024 * 1024

db = SQLAlchemy(app)

# Database Models
class Admin(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    username = db.Column(db.String(80), unique=True, nullable=False)
    password_hash = db.Column(db.String(255), nullable=False)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)

class Shipment(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    tracking_number = db.Column(db.String(20), unique=True, nullable=False)
    shipper_name = db.Column(db.String(100), nullable=False)
    shipper_address = db.Column(db.Text, nullable=False)
    receiver_name = db.Column(db.String(100), nullable=False)
    receiver_address = db.Column(db.Text, nullable=False)
    shipment_date = db.Column(db.DateTime, nullable=False)
    delivery_date = db.Column(db.DateTime, nullable=True)
    status = db.Column(db.String(50), default='Processing')
    weight = db.Column(db.Float, default=0.0)
    service_type = db.Column(db.String(50), default='Standard')
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    updated_at = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)

class TrackingHistory(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    tracking_number = db.Column(db.String(20), db.ForeignKey('shipment.tracking_number'), nullable=False)
    status = db.Column(db.String(50), nullable=False)
    location = db.Column(db.String(200), nullable=False)
    description = db.Column(db.Text, nullable=True)
    timestamp = db.Column(db.DateTime, default=datetime.utcnow)
    backcode = db.Column(db.String(50), nullable=True)

# Initialize database
with app.app_context():
    db.create_all()
    
    # Create default admin if not exists
    admin = Admin.query.filter_by(username='admin').first()
    if not admin:
        admin = Admin(
            username='admin',
            password_hash=generate_password_hash('Adedapo1997$')
        )
        db.session.add(admin)
        db.session.commit()

# Helper functions
def generate_tracking_number():
    """Generate a unique tracking number"""
    while True:
        # Format: NS-XXXXXXXX-XXXX (NS for NexusShip)
        num1 = ''.join(random.choices(string.digits, k=8))
        num2 = ''.join(random.choices(string.digits, k=4))
        tracking_num = f"NS-{num1}-{num2}"
        if not Shipment.query.filter_by(tracking_number=tracking_num).first():
            return tracking_num

def generate_backcode(status):
    """Generate backcode based on status"""
    codes = {
        'Processing': 'PROC',
        'In Transit': 'TRNS',
        'Out for Delivery': 'OUTD',
        'Delivered': 'DLVR',
        'On Hold': 'HOLD',
        'Exception': 'EXCP',
        'Delayed': 'DLYD'
    }
    prefix = codes.get(status, 'UNKN')
    return f"{prefix}-{datetime.utcnow().strftime('%Y%m%d')}-{random.randint(1000, 9999)}"

# Routes
@app.route('/')
def index():
    return render_template('index.html')

@app.route('/about')
def about():
    return render_template('about.html')

@app.route('/contact')
def contact():
    return render_template('contact.html')

@app.route('/track')
def track():
    return render_template('track.html')

@app.route('/admin')
def admin():
    if 'admin_logged_in' not in session:
        return redirect(url_for('admin_login'))
    return render_template('admin.html')

@app.route('/admin/login', methods=['GET', 'POST'])
def admin_login():
    if request.method == 'POST':
        username = request.form.get('username')
        password = request.form.get('password')
        
        admin = Admin.query.filter_by(username=username).first()
        if admin and check_password_hash(admin.password_hash, password):
            session['admin_logged_in'] = True
            session['admin_username'] = username
            return redirect(url_for('admin'))
        else:
            return render_template('admin_login.html', error='Invalid credentials')
    
    return render_template('admin_login.html')

@app.route('/admin/logout')
def admin_logout():
    session.pop('admin_logged_in', None)
    session.pop('admin_username', None)
    return redirect(url_for('admin_login'))

# API Routes
@app.route('/api/track', methods=['POST'])
def api_track():
    data = request.json
    tracking_number = data.get('tracking_number')
    
    shipment = Shipment.query.filter_by(tracking_number=tracking_number).first()
    
    if not shipment:
        return jsonify({'error': 'Tracking number not found'}), 404
    
    # Get tracking history
    history = TrackingHistory.query.filter_by(
        tracking_number=tracking_number
    ).order_by(TrackingHistory.timestamp.desc()).all()
    
    history_data = []
    for h in history:
        history_data.append({
            'status': h.status,
            'location': h.location,
            'description': h.description,
            'timestamp': h.timestamp.strftime('%Y-%m-%d %H:%M:%S'),
            'backcode': h.backcode
        })
    
    return jsonify({
        'tracking_number': shipment.tracking_number,
        'shipper_name': shipment.shipper_name,
        'shipper_address': shipment.shipper_address,
        'receiver_name': shipment.receiver_name,
        'receiver_address': shipment.receiver_address,
        'shipment_date': shipment.shipment_date.strftime('%Y-%m-%d'),
        'delivery_date': shipment.delivery_date.strftime('%Y-%m-%d') if shipment.delivery_date else None,
        'status': shipment.status,
        'weight': shipment.weight,
        'service_type': shipment.service_type,
        'history': history_data
    })

@app.route('/api/admin/generate-tracking', methods=['POST'])
def api_generate_tracking():
    if 'admin_logged_in' not in session:
        return jsonify({'error': 'Unauthorized'}), 401
    
    count = request.json.get('count', 300)
    tracking_numbers = []
    
    for _ in range(count):
        tracking_num = generate_tracking_number()
        tracking_numbers.append(tracking_num)
    
    return jsonify({'tracking_numbers': tracking_numbers})

@app.route('/api/admin/shipments', methods=['GET'])
def api_get_shipments():
    if 'admin_logged_in' not in session:
        return jsonify({'error': 'Unauthorized'}), 401
    
    shipments = Shipment.query.order_by(Shipment.created_at.desc()).all()
    shipments_data = []
    
    for s in shipments:
        shipments_data.append({
            'id': s.id,
            'tracking_number': s.tracking_number,
            'shipper_name': s.shipper_name,
            'receiver_name': s.receiver_name,
            'status': s.status,
            'shipment_date': s.shipment_date.strftime('%Y-%m-%d'),
            'delivery_date': s.delivery_date.strftime('%Y-%m-%d') if s.delivery_date else None
        })
    
    return jsonify({'shipments': shipments_data})

@app.route('/api/admin/shipment', methods=['POST'])
def api_create_shipment():
    if 'admin_logged_in' not in session:
        return jsonify({'error': 'Unauthorized'}), 401
    
    data = request.json
    
    # Validate tracking number exists
    if Shipment.query.filter_by(tracking_number=data['tracking_number']).first():
        return jsonify({'error': 'Tracking number already in use'}), 400
    
    # Parse dates
    shipment_date = datetime.strptime(data['shipment_date'], '%Y-%m-%d')
    delivery_date = None
    if data.get('delivery_date'):
        delivery_date = datetime.strptime(data['delivery_date'], '%Y-%m-%d')
    
    shipment = Shipment(
        tracking_number=data['tracking_number'],
        shipper_name=data['shipper_name'],
        shipper_address=data['shipper_address'],
        receiver_name=data['receiver_name'],
        receiver_address=data['receiver_address'],
        shipment_date=shipment_date,
        delivery_date=delivery_date,
        status=data.get('status', 'Processing'),
        weight=float(data.get('weight', 0)),
        service_type=data.get('service_type', 'Standard')
    )
    
    db.session.add(shipment)
    db.session.commit()
    
    # Add initial tracking history
    initial_history = TrackingHistory(
        tracking_number=data['tracking_number'],
        status=data.get('status', 'Processing'),
        location=data.get('location', 'Origin Facility'),
        description=f"Shipment created and processed",
        backcode=generate_backcode(data.get('status', 'Processing'))
    )
    db.session.add(initial_history)
    db.session.commit()
    
    return jsonify({'success': True, 'tracking_number': shipment.tracking_number})

@app.route('/api/admin/shipment/<int:shipment_id>', methods=['PUT'])
def api_update_shipment(shipment_id):
    if 'admin_logged_in' not in session:
        return jsonify({'error': 'Unauthorized'}), 401
    
    data = request.json
    shipment = Shipment.query.get_or_404(shipment_id)
    
    # Update basic info
    if 'shipper_name' in data:
        shipment.shipper_name = data['shipper_name']
    if 'shipper_address' in data:
        shipment.shipper_address = data['shipper_address']
    if 'receiver_name' in data:
        shipment.receiver_name = data['receiver_name']
    if 'receiver_address' in data:
        shipment.receiver_address = data['receiver_address']
    
    if 'shipment_date' in data:
        shipment.shipment_date = datetime.strptime(data['shipment_date'], '%Y-%m-%d')
    if 'delivery_date' in data:
        if data['delivery_date']:
            shipment.delivery_date = datetime.strptime(data['delivery_date'], '%Y-%m-%d')
        else:
            shipment.delivery_date = None
    
    if 'status' in data:
        shipment.status = data['status']
        
        # Add tracking history entry
        if 'location' in data:
            new_history = TrackingHistory(
                tracking_number=shipment.tracking_number,
                status=data['status'],
                location=data['location'],
                description=data.get('description', f"Status updated to {data['status']}"),
                backcode=generate_backcode(data['status'])
            )
            db.session.add(new_history)
    
    if 'weight' in data:
        shipment.weight = float(data['weight'])
    if 'service_type' in data:
        shipment.service_type = data['service_type']
    
    shipment.updated_at = datetime.utcnow()
    db.session.commit()
    
    return jsonify({'success': True})

@app.route('/api/admin/shipment/<int:shipment_id>', methods=['DELETE'])
def api_delete_shipment(shipment_id):
    if 'admin_logged_in' not in session:
        return jsonify({'error': 'Unauthorized'}), 401
    
    shipment = Shipment.query.get_or_404(shipment_id)
    
    # Delete tracking history first
    TrackingHistory.query.filter_by(tracking_number=shipment.tracking_number).delete()
    
    db.session.delete(shipment)
    db.session.commit()
    
    return jsonify({'success': True})

@app.route('/api/admin/bulk-update-status', methods=['POST'])
def api_bulk_update_status():
    if 'admin_logged_in' not in session:
        return jsonify({'error': 'Unauthorized'}), 401
    
    data = request.json
    tracking_numbers = data.get('tracking_numbers', [])
    status = data.get('status')
    location = data.get('location', 'Updated')
    description = data.get('description', f"Status updated to {status}")
    
    for tracking_num in tracking_numbers:
        shipment = Shipment.query.filter_by(tracking_number=tracking_num).first()
        if shipment:
            shipment.status = status
            
            # Add tracking history
            new_history = TrackingHistory(
                tracking_number=tracking_num,
                status=status,
                location=location,
                description=description,
                backcode=generate_backcode(status)
            )
            db.session.add(new_history)
    
    db.session.commit()
    
    return jsonify({'success': True, 'updated': len(tracking_numbers)})

@app.route('/api/receipt/<tracking_number>')
def api_receipt(tracking_number):
    shipment = Shipment.query.filter_by(tracking_number=tracking_number).first()
    
    if not shipment:
        return jsonify({'error': 'Tracking number not found'}), 404
    
    # Get tracking history
    history = TrackingHistory.query.filter_by(
        tracking_number=tracking_number
    ).order_by(TrackingHistory.timestamp.desc()).all()
    
    history_data = []
    for h in history:
        history_data.append({
            'status': h.status,
            'location': h.location,
            'description': h.description,
            'timestamp': h.timestamp.strftime('%Y-%m-%d %H:%M:%S'),
            'backcode': h.backcode
        })
    
    return jsonify({
        'tracking_number': shipment.tracking_number,
        'shipper_name': shipment.shipper_name,
        'shipper_address': shipment.shipper_address,
        'receiver_name': shipment.receiver_name,
        'receiver_address': shipment.receiver_address,
        'shipment_date': shipment.shipment_date.strftime('%Y-%m-%d'),
        'delivery_date': shipment.delivery_date.strftime('%Y-%m-%d') if shipment.delivery_date else None,
        'status': shipment.status,
        'weight': shipment.weight,
        'service_type': shipment.service_type,
        'history': history_data,
        'created_at': shipment.created_at.strftime('%Y-%m-%d %H:%M:%S')
    })

@app.route('/api/receipt/pdf/<tracking_number>')
def api_receipt_pdf(tracking_number):
    shipment = Shipment.query.filter_by(tracking_number=tracking_number).first()
    
    if not shipment:
        return "Shipment not found", 404
    
    # Get tracking history
    history = TrackingHistory.query.filter_by(
        tracking_number=tracking_number
    ).order_by(TrackingHistory.timestamp.desc()).all()
    
    # Create PDF
    buffer = BytesIO()
    doc = SimpleDocTemplate(buffer, pagesize=A4)
    elements = []
    styles = getSampleStyleSheet()
    
    # Custom styles
    title_style = ParagraphStyle(
        'CustomTitle',
        parent=styles['Heading1'],
        fontSize=24,
        textColor=colors.HexColor('#4D148C'),
        alignment=TA_CENTER,
        spaceAfter=30
    )
    
    # Header
    elements.append(Paragraph("NexusShip", title_style))
    elements.append(Paragraph("Shipping Receipt", styles['Heading2']))
    elements.append(Spacer(1, 20))
    
    # Generate QR Code
    qr = qrcode.QRCode(version=1, box_size=10, border=5)
    qr.add_data(shipment.tracking_number)
    qr.make(fit=True)
    qr_image = qr.make_image(fill_color="black", back_color="white")
    
    # Save QR code to temporary buffer
    qr_buffer = BytesIO()
    qr_image.save(qr_buffer, format='PNG')
    qr_buffer.seek(0)
    
    # Add tracking number and QR code
    tracking_data = [
        ['Tracking Number:', shipment.tracking_number],
        ['Status:', shipment.status],
        ['Service Type:', shipment.service_type]
    ]
    
    tracking_table = Table(tracking_data, colWidths=[150, 300])
    tracking_table.setStyle(TableStyle([
        ('BACKGROUND', (0, 0), (0, 2), colors.HexColor('#4D148C')),
        ('TEXTCOLOR', (0, 0), (0, 2), colors.whitesmoke),
        ('ALIGN', (0, 0), (-1, -1), 'LEFT'),
        ('FONTNAME', (0, 0), (-1, -1), 'Helvetica'),
        ('FONTSIZE', (0, 0), (-1, -1), 12),
        ('BOTTOMPADDING', (0, 0), (-1, -1), 12),
        ('BACKGROUND', (1, 0), (1, 2), colors.beige),
        ('GRID', (0, 0), (-1, -1), 1, colors.black)
    ]))
    
    elements.append(tracking_table)
    elements.append(Spacer(1, 20))
    
    # Shipper and Receiver Information
    elements.append(Paragraph("Shipper Information", styles['Heading3']))
    shipper_data = [
        ['Name:', shipment.shipper_name],
        ['Address:', shipment.shipper_address.replace('\n', ', ')]
    ]
    
    shipper_table = Table(shipper_data, colWidths=[100, 350])
    shipper_table.setStyle(TableStyle([
        ('BACKGROUND', (0, 0), (0, -1), colors.HexColor('#666666')),
        ('TEXTCOLOR', (0, 0), (0, -1), colors.whitesmoke),
        ('ALIGN', (0, 0), (-1, -1), 'LEFT'),
        ('FONTNAME', (0, 0), (-1, -1), 'Helvetica'),
        ('FONTSIZE', (0, 0), (-1, -1), 10),
        ('BOTTOMPADDING', (0, 0), (-1, -1), 10),
        ('BACKGROUND', (1, 0), (1, -1), colors.lightgrey),
        ('GRID', (0, 0), (-1, -1), 1, colors.black)
    ]))
    
    elements.append(shipper_table)
    elements.append(Spacer(1, 20))
    
    elements.append(Paragraph("Receiver Information", styles['Heading3']))
    receiver_data = [
        ['Name:', shipment.receiver_name],
        ['Address:', shipment.receiver_address.replace('\n', ', ')]
    ]
    
    receiver_table = Table(receiver_data, colWidths=[100, 350])
    receiver_table.setStyle(TableStyle([
        ('BACKGROUND', (0, 0), (0, -1), colors.HexColor('#666666')),
        ('TEXTCOLOR', (0, 0), (0, -1), colors.whitesmoke),
        ('ALIGN', (0, 0), (-1, -1), 'LEFT'),
        ('FONTNAME', (0, 0), (-1, -1), 'Helvetica'),
        ('FONTSIZE', (0, 0), (-1, -1), 10),
        ('BOTTOMPADDING', (0, 0), (-1, -1), 10),
        ('BACKGROUND', (1, 0), (1, -1), colors.lightgrey),
        ('GRID', (0, 0), (-1, -1), 1, colors.black)
    ]))
    
    elements.append(receiver_table)
    elements.append(Spacer(1, 20))
    
    # Dates
    dates_data = [
        ['Shipment Date:', shipment.shipment_date.strftime('%Y-%m-%d')],
        ['Expected Delivery:', shipment.delivery_date.strftime('%Y-%m-%d') if shipment.delivery_date else 'Pending']
    ]
    
    dates_table = Table(dates_data, colWidths=[150, 300])
    dates_table.setStyle(TableStyle([
        ('BACKGROUND', (0, 0), (0, 1), colors.HexColor('#4D148C')),
        ('TEXTCOLOR', (0, 0), (0, 1), colors.whitesmoke),
        ('ALIGN', (0, 0), (-1, -1), 'LEFT'),
        ('FONTNAME', (0, 0), (-1, -1), 'Helvetica'),
        ('FONTSIZE', (0, 0), (-1, -1), 12),
        ('BOTTOMPADDING', (0, 0), (-1, -1), 12),
        ('BACKGROUND', (1, 0), (1, 1), colors.beige),
        ('GRID', (0, 0), (-1, -1), 1, colors.black)
    ]))
    
    elements.append(dates_table)
    elements.append(Spacer(1, 20))
    
    # Tracking History
    elements.append(Paragraph("Tracking History", styles['Heading3']))
    
    history_rows = [['Status', 'Location', 'Description', 'Date/Time', 'Backcode']]
    for h in history:
        history_rows.append([
            h.status,
            h.location,
            h.description,
            h.timestamp.strftime('%Y-%m-%d %H:%M'),
            h.backcode
        ])
    
    history_table = Table(history_rows, colWidths=[80, 120, 150, 120, 80])
    history_table.setStyle(TableStyle([
        ('BACKGROUND', (0, 0), (-1, 0), colors.HexColor('#4D148C')),
        ('TEXTCOLOR', (0, 0), (-1, 0), colors.whitesmoke),
        ('ALIGN', (0, 0), (-1, -1), 'LEFT'),
        ('FONTNAME', (0, 0), (-1, -1), 'Helvetica'),
        ('FONTSIZE', (0, 0), (-1, -1), 9),
        ('TOPPADDING', (0, 0), (-1, -1), 8),
        ('BOTTOMPADDING', (0, 0), (-1, -1), 8),
        ('GRID', (0, 0), (-1, -1), 1, colors.black),
        ('ROWBACKGROUNDS', (0, 1), (-1, -1), [colors.white, colors.lightgrey])
    ]))
    
    elements.append(history_table)
    elements.append(Spacer(1, 30))
    
    # Footer
    footer_text = "Thank you for choosing NexusShip. For inquiries, contact us at support@nexusship.com"
    elements.append(Paragraph(footer_text, ParagraphStyle('Footer', alignment=TA_CENTER, fontSize=10)))
    
    # Build PDF
    doc.build(elements)
    buffer.seek(0)
    
    return send_file(
        buffer,
        as_attachment=True,
        download_name=f'receipt_{tracking_number}.pdf',
        mimetype='application/pdf'
    )

if __name__ == '__main__':
    app.run(debug=True, host='0.0.0.0', port=3000)